//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System;
using System.Collections.Generic;
using System.Configuration;
using System.Reflection;
using System.Text;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Factory;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Properties;

namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection
{
	/// <summary>
	/// Represents the base class from which all implementations of injection factories must derive.
	/// </summary>
	/// <typeparam name="TObjectToCreate">The object type to create.</typeparam>
	/// <typeparam name="TObjectConfiguration">The object configuration for the object to create.</typeparam>
	/// <remarks>
	/// The injection factory inspects <typeparamref name="TObjectConfiguration"/> and looks for the <see cref="ConstructorParameterAttribute"/>s 
	/// to get the values to inject into the <typeparamref name="TObjectToCreate"/> constructor when creating the instance to return.
	/// </remarks>
	public class InjectionFactory<TObjectToCreate, TObjectConfiguration> : ObjectFactoryBase<TObjectToCreate, TObjectConfiguration>
	{
        public virtual TObjectToCreate Create()
        {
            return Create(default(TObjectConfiguration));
        }

        public override TObjectToCreate Create(TObjectConfiguration objectConfiguration, IConfigurationSource configurationSource)
        {
            return Create(GetObjectType(objectConfiguration, configurationSource), GetConfiguration(objectConfiguration, configurationSource), configurationSource);
        }

		/// <summary>
		/// Gets the type of object that the factory will create.
		/// </summary>
		/// <param name="objectConfiguration">The configuration for the object to create.</param>
		/// <param name="configurationSource">An <see cref="IConfigurationSource"/> object.</param>
		/// <returns>The object type to create. This defaults to the <typeparamref name="TObjectToCreate"/> type.</returns>
		protected virtual Type GetObjectType(TObjectConfiguration objectConfiguration, IConfigurationSource configurationSource)
		{
            return typeof(TObjectToCreate);
		}

        protected virtual TObjectConfiguration GetConfiguration(TObjectConfiguration objectConfiguration, IConfigurationSource configurationSource)
        {
            return objectConfiguration;
        }

		/// <summary>
		/// Creates the object instance.
		/// </summary>
		/// <param name="objectType">The type of object to create.</param>
		/// <param name="objectConfiguration">The configuration for the object to create.</param>
		/// <param name="configurationSource">An <see cref="IConfigurationSource"/> object.</param>
		/// <returns>The configured object instance.</returns>
		protected virtual TObjectToCreate Create(Type objectType, TObjectConfiguration objectConfiguration, IConfigurationSource configurationSource)
		{
            ConstructableObject<TObjectToCreate, TObjectConfiguration> objectToCreate = new ConstructableObject<TObjectToCreate, TObjectConfiguration>(objectType, objectConfiguration, configurationSource);
            TObjectToCreate result = objectToCreate.Create();

			return result;
		}
	}
}
